<?php

namespace App\Services\Common;

use App\Domains\Engine\Enums\EngineEnum;
use App\Enums\Introduction;
use App\Helpers\Classes\Helper;
use App\Models\Common\Menu;
use App\Models\Extension;
use App\Models\Integration\Integration;
use App\Models\Plan;
use App\Models\Setting;
use App\Models\SettingTwo;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;

class MenuService
{
    public const MENU_KEY = 'dynamic_menu_key';

    public function cacheKey(): string
    {
        return Auth::user()?->isAdmin() || Auth::user()?->isSuperAdmin() . get_theme() . '-navbar-menu';
    }

    public function regenerate(): array
    {
        cache()->forget(self::MENU_KEY);

        cache()->forget($this->cacheKey());

        return $this->generate(false);
    }

    public function boltMenu(): array
    {
        $data = collect($this->generate(false))
            ->where('bolt_menu', true);

        $array = [];

        foreach ($data as $key => $value) {
            $array[$key] = [
                'background' => $value['bolt_background'],
                'foreground' => $value['bolt_foreground'],
            ];
        }

        return $array;
    }

    public function generate(bool $active = true): array
    {
        return cache()->rememberForever(self::MENU_KEY, function () use ($active) {
            $items = Menu::query()
                ->with('children')
                ->withCount('children')
                ->whereNull('parent_id')
                ->when($active, function ($query) {
                    $query->where('is_active', true);
                })
                ->orderBy('order', 'asc')
                ->get();

            return $this->merge($items);
        });

    }

    public function subMenuOrderUpdate(array $data)
    {

        $order = 0;

        $lastParent = 0;

        foreach ($data as $key => $value) {
            if ($value != $lastParent) {
                $order = 0;
            } else {
                $order = $order + 1;
            }

            Menu::query()
                ->where('id', $key)
                ->update([
                    'order'     => $order,
                    'parent_id' => $value,
                ]);

            $lastParent = $value;
        }
    }

    public function parentMenuOrderUpdate(array $data): void
    {
        foreach ($data as $key => $value) {
            Menu::query()
                ->where('id', $value)
                ->update([
                    'parent_id' => null,
                    'order'     => $key,
                ]);
        }
    }

    public function merge($items, $else = true): array
    {
        $data = [];

        $staticData = $this->data();

        foreach ($items as $item) {

            if (isset($staticData[$item['key']])) {

                $data[$item['key']] = array_merge($staticData[$item['key']], $item->toArray());

                if ($item->parent_id == null && $else) {
                    $children = $item->getAttribute('children');

                    $data[$item['key']]['children'] = $this->merge($children);
                }

            } elseif ($else) {
                $data[$item['key']] = array_merge($item->toArray(), [
                    'active_condition' => false,
                    'show_condition'   => true,
                    'extension'        => false,
                ]);
                if ($item->parent_id == null && $else) {
                    $children = $item->getAttribute('children');

                    $data[$item['key']]['children'] = $this->merge($children);
                }
            }
        }

        return $data;
    }

    public function data(): array
    {
        $admin = Auth::user()?->isAdmin();

        $setting = Setting::query()->first();
        $settings_two = SettingTwo::query()->first();

        $menu = [
            'user_label' => [
                'parent_key'       => null,
                'key'              => 'user_label',
                'route'            => null,
                'label'            => 'User',
                'icon'             => null,
                'svg'              => null,
                'order'            => 1,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'label',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
            ],
            'dashboard' => [
                'parent_key'       => null,
                'key'              => 'dashboard',
                'route'            => 'dashboard.user.index',
                'label'            => 'Dashboard',
                'icon'             => 'tabler-layout-2',
                'svg'              => null,
                'order'            => 2,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.index',
                ],
                'show_condition' => true,
            ],
            'ext_chat_bot' => [
                'parent_key'       => null,
                'key'              => 'ext_chat_bot',
                'route'            => 'dashboard.chatbot.index',
                'label'            => 'AI Bots',
                'icon'             => 'tabler-message-chatbot',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.chatbot.*',
                ],
                'show_condition' => Route::has('dashboard.chatbot.index'),
            ],
            'documents' => [
                'parent_key'       => null,
                'key'              => 'documents',
                'route'            => 'dashboard.user.openai.documents.all',
                'label'            => 'Documents',
                'icon'             => 'tabler-archive',
                'svg'              => null,
                'order'            => 3,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.documents.*',
                ],
                'show_condition' => true,
            ],
            'ai_editor' => [
                'parent_key'       => null,
                'key'              => 'ai_editor',
                'route'            => 'dashboard.user.generator.index',
                'label'            => 'AI Editor',
                'icon'             => 'tabler-notebook',
                'svg'              => null,
                'order'            => 4,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.generator.index',
                ],
                'show_condition' => (bool) Helper::setting('feature_ai_advanced_editor', null, $setting),
            ],
            'ai_writer' => [
                'parent_key'       => null,
                'key'              => 'ai_writer',
                'route'            => 'dashboard.user.openai.list',
                'label'            => 'AI Writer',
                'data-name'        => Introduction::AI_WRITER,
                'icon'             => 'tabler-notes',
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.list', 'dashboard.user.openai.generator.*', 'dashboard.user.openai.generator.workbook',
                ],
                'show_condition' => (bool) Helper::setting('feature_ai_writer', null, $setting),
            ],
            'ai_video' => [
                'parent_key'       => null,
                'key'              => 'ai_video',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_video',
                'label'            => 'AI Video',
                'icon'             => 'tabler-video',
                'svg'              => null,
                'order'            => 6,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::settingTwo('feature_ai_video', null, $settings_two),
            ],
            'ai_image_generator' => [
                'parent_key'       => null,
                'key'              => 'ai_image_generator',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_image_generator',
                'label'            => 'AI Image',
                'data-name'        => Introduction::AI_IMAGE,
                'icon'             => 'tabler-photo',
                'svg'              => null,
                'order'            => 7,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_image', null, $setting),
            ],
            'ai_article_wizard' => [
                'parent_key'       => null,
                'key'              => 'ai_article_wizard',
                'route'            => 'dashboard.user.openai.articlewizard.new',
                'route_slug'       => 'ai_article_wizard',
                'label'            => 'AI Article Wizard',
                'icon'             => 'tabler-ad-2',
                'svg'              => null,
                'order'            => 8,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.articlewizard.new',
                ],
                'show_condition' => (bool) Helper::setting('feature_ai_article_wizard', null, $setting),
            ],
            'ai_pdf' => [
                'parent_key'       => null,
                'key'              => 'ai_pdf',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_pdf',
                'label'            => 'AI File Chat',
                'data-name'        => Introduction::AI_PDF,
                'icon'             => 'tabler-file-pencil',
                'svg'              => null,
                'order'            => 9,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_pdf', null, $setting),
            ],
            'ai_vision' => [
                'parent_key'       => null,
                'key'              => 'ai_vision',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_vision',
                'label'            => 'AI Vision',
                'icon'             => 'tabler-scan-eye',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_vision', null, $setting),
            ],
            'ai_rewriter' => [
                'parent_key'       => null,
                'key'              => 'ai_rewriter',
                'route'            => 'dashboard.user.openai.rewriter',
                'route_slug'       => 'ai_rewriter',
                'label'            => 'AI ReWriter',
                'icon'             => 'tabler-ballpen',
                'svg'              => null,
                'order'            => 11,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.rewriter', 'dashboard.user.openai.rewriter.*',
                ],
                'show_condition' => (bool) Helper::setting('feature_ai_rewriter', null, $setting),
            ],
            'ai_chat_image' => [
                'parent_key'       => null,
                'key'              => 'ai_chat_image',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_chat_image',
                'label'            => 'AI Chat Image',
                'icon'             => 'tabler-photo',
                'svg'              => null,
                'order'            => 12,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => route('dashboard.user.openai.generator.workbook', 'ai_chat_image') === url()->current(),
                'show_condition'   => (bool) Helper::setting('feature_ai_chat_image', null, $setting),
            ],
            'ai_chat_all' => [
                'parent_key'       => null,
                'key'              => 'ai_chat_all',
                'route'            => 'dashboard.user.openai.chat.list',
                'route_slug'       => null,
                'label'            => 'AI Chat',
                'icon'             => 'tabler-message-dots',
                'svg'              => null,
                'order'            => 13,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.openai.chat.*',
                ],
                'show_condition' => (bool) Helper::setting('feature_ai_chat', null, $setting),
            ],
            'ai_code_generator' => [
                'parent_key'       => null,
                'key'              => 'ai_code_generator',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_code_generator',
                'label'            => 'AI Code',
                'data-name'        => Introduction::AI_CODE,
                'icon'             => 'tabler-terminal-2',
                'svg'              => null,
                'order'            => 14,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_code', null, $setting),
            ],
            'ai_youtube' => [
                'parent_key'       => null,
                'key'              => 'ai_youtube',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_youtube',
                'label'            => 'AI YouTube',
                'icon'             => 'tabler-brand-youtube',
                'svg'              => null,
                'order'            => 15,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_youtube', null, $setting),
            ],
            'ai_rss' => [
                'parent_key'       => null,
                'key'              => 'ai_rss',
                'route'            => 'dashboard.user.openai.generator.workbook',
                'route_slug'       => 'ai_rss',
                'label'            => 'AI RSS',
                'icon'             => 'tabler-rss',
                'svg'              => null,
                'order'            => 15,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_rss', null, $setting),
            ],
            'ai_speech_to_text' => [
                'parent_key'       => null,
                'key'              => 'ai_speech_to_text',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_speech_to_text',
                'label'            => 'AI Speech to Text',
                'icon'             => 'tabler-microphone',
                'svg'              => null,
                'order'            => 16,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_speech_to_text', null, $setting),
            ],
            'ai_voiceover' => [
                'parent_key'       => null,
                'key'              => 'ai_voiceover',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_voiceover',
                'label'            => 'AI Voiceover',
                'icon'             => 'tabler-volume',
                'svg'              => null,
                'order'            => 17,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => (bool) Helper::setting('feature_ai_voiceover', null, $setting),
            ],
            'ai_voice_isolator' => [
                'parent_key'       => null,
                'key'              => 'ai_voice_isolator',
                'route'            => 'dashboard.user.openai.generator',
                'route_slug'       => 'ai_voice_isolator',
                'label'            => 'AI Voice Isolator',
                'icon'             => 'tabler-ear-scan',
                'svg'              => null,
                'order'            => 18,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => setting('ai_voice_isolator', '1') == '1'
                    && file_exists(base_path('resources/views/default/panel/user/openai/components/generator_voice_isolator.blade.php')),
            ],
            'ai_voiceover_clone' => [
                'parent_key'       => null,
                'key'              => 'ai_voiceover_clone',
                'route'            => 'dashboard.user.voice.index',
                'route_slug'       => null,
                'label'            => 'AI Voice Clone',
                'icon'             => 'tabler-microphone-2',
                'svg'              => null,
                'order'            => 18,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.voice.*',
                ],
                'show_condition' => Helper::setting('feature_ai_voice_clone', null, $setting) && Helper::settingTwo('elevenlabs_api_key', null, $settings_two),
            ],
            'team_menu' => [
                'parent_key'       => null,
                'key'              => 'team_menu',
                'route'            => 'dashboard.user.team.index',
                'route_slug'       => null,
                'label'            => 'Team',
                'icon'             => 'tabler-user-plus',
                'svg'              => null,
                'order'            => 19,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.team.*',
                ],
                'show_condition' => $this->showTeamFunctionality(),
            ],
            'brand_voice' => [
                'parent_key'       => null,
                'key'              => 'brand_voice',
                'route'            => 'dashboard.user.brand.index',
                'route_slug'       => null,
                'label'            => 'Brand Voice',
                'icon'             => 'tabler-brand-trello',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => true,
            ],
            'advanced_image' => [
                'parent_key'       => null,
                'key'              => 'advanced_image',
                'route'            => 'dashboard.user.advanced-image.index',
                'route_slug'       => null,
                'label'            => 'Advanced Image Editor',
                'icon'             => 'tabler-photo-edit',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => Route::has('dashboard.user.advanced-image.index'),
            ],
            'ai_avatar' => [
                'parent_key'       => null,
                'key'              => 'ai_avatar',
                'route'            => 'dashboard.user.ai-avatar.index',
                'route_slug'       => null,
                'label'            => 'AI Avatar',
                'icon'             => 'tabler-slideshow',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => Route::has('dashboard.user.ai-avatar.index'),
            ],
            'ai_avatar_pro' => [
                'parent_key'       => null,
                'key'              => 'ai_avatar_pro',
                'route'            => 'dashboard.user.ai-avatar-pro.index',
                'route_slug'       => null,
                'label'            => 'AI Avatar Pro',
                'icon'             => 'tabler-camera-star',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => Route::has('dashboard.user.ai-avatar-pro.index'),
            ],
            'ai_fall' => [
                'parent_key'       => null,
                'key'              => 'ai_fall',
                'route'            => 'dashboard.user.fall-video.index',
                'route_slug'       => null,
                'label'            => 'AI Video Pro',
                'icon'             => 'tabler-video',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => false,
                'show_condition'   => Route::has('dashboard.admin.settings.fal-ai'),
            ],
            'ai_product_shot' => [
                'parent_key'       => null,
                'key'              => 'ai_product_shot',
                'route'            => 'dashboard.user.ai-product-shot.index',
                'route_slug'       => null,
                'label'            => 'AI Product Photography',
                'icon'             => 'tabler-photo-star',
                'svg'              => null,
                'order'            => 20,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => null,
                'show_condition'   => Route::has('dashboard.user.ai-product-shot.index'),
            ],
            'api_keys' => [
                'parent_key'       => null,
                'key'              => 'api_keys',
                'route'            => 'dashboard.user.apikeys.index',
                'route_slug'       => null,
                'label'            => 'API Keys',
                'icon'             => 'tabler-key',
                'svg'              => null,
                'order'            => 21,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.apikeys.*',
                ],
                'show_condition' => Helper::appIsDemo() || Helper::setting('user_api_option', null, $setting),
            ],
            'affiliates' => [
                'parent_key'       => null,
                'key'              => 'affiliates',
                'route'            => 'dashboard.user.affiliates.index',
                'route_slug'       => null,
                'label'            => 'Affiliates',
                'icon'             => 'tabler-currency-dollar',
                'svg'              => null,
                'order'            => 22,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => Helper::setting('feature_affilates', null, $setting),
            ],
            'support' => [
                'parent_key'       => null,
                'key'              => 'support',
                'route'            => 'dashboard.support.list',
                'route_slug'       => null,
                'label'            => 'Support',
                'icon'             => 'tabler-lifebuoy',
                'svg'              => null,
                'order'            => 23,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => true,
            ],
            'integration' => [
                'parent_key'       => null,
                'key'              => 'integration',
                'route'            => 'dashboard.user.integration.index',
                'route_slug'       => null,
                'label'            => 'Integration',
                'icon'             => 'tabler-webhook',
                'svg'              => null,
                'order'            => 24,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.user.integration.*',
                ],
                'show_condition' => Integration::query()->whereHas('hasExtension')->count(),
            ],
            'divider_one' => [
                'parent_key'       => null,
                'key'              => 'divider_one',
                'route'            => null,
                'label'            => null,
                'icon'             => null,
                'svg'              => null,
                'order'            => 25,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'divider',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
            ],
            'links' => [
                'parent_key'       => null,
                'key'              => 'links',
                'route'            => null,
                'label'            => 'Links',
                'icon'             => null,
                'svg'              => null,
                'order'            => 26,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'label',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
            ],
            'favorites' => [
                'parent_key'       => null,
                'key'              => 'favorites',
                'route'            => 'dashboard.user.openai.list',
                'route_slug'       => 'filter=favorite',
                'label'            => 'Favorites',
                'icon'             => null,
                'svg'              => null,
                'order'            => 27,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
                'letter_icon'      => 1,
                'letter_icon_bg'   => 'bg-[#7A8193] text-white',
            ],
            'workbook' => [
                'parent_key'       => null,
                'key'              => 'workbook',
                'route'            => 'dashboard.user.openai.documents.all',
                'route_slug'       => '?filter=favorites',
                'label'            => 'Workbook',
                'icon'             => null,
                'svg'              => null,
                'order'            => 28,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
                'letter_icon'      => 1,
                'letter_icon_bg'   => 'bg-[#658C8E] text-white',
            ],
            'divider_two' => [
                'parent_key'       => null,
                'key'              => 'divider_two',
                'route'            => null,
                'label'            => null,
                'icon'             => null,
                'svg'              => null,
                'order'            => 29,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'divider',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'admin_label' => [
                'parent_key'       => null,
                'key'              => 'admin_label',
                'route'            => null,
                'label'            => 'Admin',
                'icon'             => null,
                'svg'              => null,
                'order'            => 30,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'label',
                'extension'        => null,
                'active_condition' => false,
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'admin_dashboard' => [
                'parent_key'       => null,
                'key'              => 'admin_dashboard',
                'route'            => 'dashboard.admin.index',
                'label'            => 'Dashboard',
                'icon'             => 'tabler-layout-2',
                'svg'              => null,
                'order'            => 31,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'marketplace' => [
                'parent_key'       => null,
                'key'              => 'marketplace',
                'route'            => 'dashboard.admin.marketplace.index',
                'label'            => 'Marketplace',
                'icon'             => 'tabler-building-store',
                'svg'              => null,
                'order'            => 32,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.marketplace.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'themes' => [
                'parent_key'       => null,
                'key'              => 'themes',
                'route'            => 'dashboard.admin.themes.index',
                'label'            => 'Themes',
                'icon'             => 'tabler-palette',
                'svg'              => null,
                'order'            => 33,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.themes.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'user_management' => [
                'parent_key'       => null,
                'key'              => 'user_management',
                'route'            => 'dashboard.admin.users.index',
                'label'            => 'User Management',
                'icon'             => 'tabler-users',
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'user_list' => [
                'parent_key'       => 'user_management',
                'key'              => 'user_list',
                'route'            => 'dashboard.admin.users.index',
                'label'            => 'Users List',
                'icon'             => null,
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'user_activity' => [
                'parent_key'       => 'user_management',
                'key'              => 'user_activity',
                'route'            => 'dashboard.admin.users.activity',
                'label'            => 'Users Activities',
                'icon'             => null,
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.activity',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'user_dashboard' => [
                'parent_key'       => 'user_management',
                'key'              => 'user_dashboard',
                'route'            => 'dashboard.admin.users.dashboard',
                'label'            => 'Users Dashboard',
                'icon'             => null,
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.dashboard',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'user_deletion' => [
                'parent_key'       => 'user_management',
                'key'              => 'user_deletion',
                'route'            => 'dashboard.admin.users.deletion.reqs',
                'label'            => 'User Deletion Requests',
                'icon'             => null,
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.deletion.reqs',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'user_permission' => [
                'parent_key'       => 'user_management',
                'key'              => 'user_permission',
                'route'            => 'dashboard.admin.users.permissions',
                'label'            => 'User Permissions',
                'icon'             => null,
                'svg'              => null,
                'order'            => 34,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.users.permissions',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'announcements' => [
                'parent_key'       => null,
                'key'              => 'announcements',
                'route'            => 'dashboard.admin.announcements.index',
                'label'            => 'Announcements',
                'icon'             => 'tabler-speakerphone',
                'svg'              => null,
                'order'            => 35,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.announcements.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'google_adsense' => [
                'parent_key'       => null,
                'key'              => 'google_adsense',
                'route'            => 'dashboard.admin.ads.index',
                'label'            => 'Google adsense',
                'icon'             => 'tabler-ad-circle',
                'svg'              => null,
                'order'            => 35,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.ads.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'support_requests' => [
                'parent_key'       => null,
                'key'              => 'support_requests',
                'route'            => 'dashboard.support.list',
                'label'            => 'Support Requests',
                'icon'             => 'tabler-lifebuoy',
                'svg'              => null,
                'order'            => 36,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.support.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'templates' => [
                'parent_key'       => null,
                'key'              => 'templates',
                'route'            => 'dashboard.admin.openai.list',
                'label'            => 'Templates',
                'icon'             => 'tabler-list-details',
                'svg'              => null,
                'order'            => 37,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => null,
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'built_in_templates' => [
                'parent_key'       => 'templates',
                'key'              => 'built_in_templates',
                'route'            => 'dashboard.admin.openai.list',
                'label'            => 'Built-in Templates',
                'icon'             => null,
                'svg'              => null,
                'order'            => 38,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.openai.list',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'custom_templates' => [
                'parent_key'       => 'templates',
                'key'              => 'custom_templates',
                'route'            => 'dashboard.admin.openai.custom.list',
                'label'            => 'Custom Templates',
                'icon'             => null,
                'svg'              => null,
                'order'            => 39,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.openai.custom.list',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'ai_writer_categories' => [
                'parent_key'       => 'templates',
                'key'              => 'ai_writer_categories',
                'route'            => 'dashboard.admin.openai.categories.list',
                'label'            => 'AI Writer Categories',
                'icon'             => null,
                'svg'              => null,
                'order'            => 40,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.openai.categories.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'chat_settings' => [
                'parent_key'       => null,
                'key'              => 'chat_settings',
                'route'            => 'dashboard.admin.openai.chat.category',
                'label'            => 'Chat Settings',
                'icon'             => 'tabler-message-circle',
                'svg'              => null,
                'order'            => 41,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.chatbot.*', 'dashboard.admin.openai.chat.list',
                    'dashboard.admin.openai.chat.addOrUpdate', 'dashboard.admin.openai.chat.category',
                    'dashboard.admin.openai.chat.addOrUpdateCategory', 'dashboard.admin.ai-chat-model.index',
                    'dashboard.admin.ai-assistant.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'chat_categories' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'chat_categories',
                'route'            => 'dashboard.admin.openai.chat.category',
                'label'            => 'Chat Categories',
                'icon'             => null,
                'svg'              => null,
                'order'            => 42,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.openai.chat.category.*', 'dashboard.admin.openai.chat.category',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'chat_templates' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'chat_templates',
                'route'            => 'dashboard.admin.openai.chat.list',
                'label'            => 'Chat Templates',
                'icon'             => null,
                'svg'              => null,
                'order'            => 43,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.openai.chat.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'chatbot_training' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'chatbot_training',
                'route'            => 'dashboard.admin.chatbot.index',
                'label'            => 'Chatbot Training',
                'icon'             => null,
                'svg'              => null,
                'order'            => 44,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.chatbot.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'ai_assistant' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'ai_assistant',
                'route'            => 'dashboard.admin.ai-assistant.index',
                'label'            => 'Assistant Training',
                'icon'             => null,
                'svg'              => null,
                'order'            => 44,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.ai-assistant.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'ai_chat_models' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'ai_chat_models',
                'route'            => 'dashboard.admin.ai-chat-model.index',
                'label'            => 'AI Chat Models',
                'icon'             => null,
                'svg'              => null,
                'order'            => 44,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.ai-chat-model.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'floating_chat_settings' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'floating_chat_settings',
                'route'            => 'dashboard.admin.chatbot.setting',
                'label'            => 'Floating Chat Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 45,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.chatbot.setting',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'external_chat_settings' => [
                'parent_key'       => 'chat_settings',
                'key'              => 'external_chat_settings',
                'route'            => 'dashboard.admin.chatbot.external_settings',
                'label'            => 'External Chat Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 46,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.chatbot.external_settings',
                ],
                'show_condition' => false,
                'is_admin'       => true,
            ],
            'frontend' => [
                'parent_key'       => null,
                'key'              => 'frontend',
                'route'            => 'dashboard.admin.frontend.settings',
                'label'            => 'Frontend',
                'icon'             => 'tabler-device-laptop',
                'svg'              => null,
                'order'            => 47,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.testimonials.*', 'dashboard.admin.frontend.authsettings', 'dashboard.admin.frontend.settings', 'dashboard.admin.frontend.faq.*', 'dashboard.admin.frontend.tools.*', 'dashboard.admin.frontend.tools.*', 'dashboard.admin.frontend.future.*', 'dashboard.admin.frontend.whois.*', 'dashboard.admin.frontend.generatorlist.*', 'dashboard.admin.clients.*', 'dashboard.admin.howitWorks.*', 'dashboard.admin.whois.*', 'dashboard.admin.frontend.menusettings', 'dashboard.admin.frontend.sectionsettings',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'frontend_settings' => [
                'parent_key'       => 'frontend',
                'key'              => 'frontend_settings',
                'route'            => 'dashboard.admin.frontend.settings',
                'label'            => 'Frontend Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 48,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.settings',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'frontend_section_settings' => [
                'parent_key'       => 'frontend',
                'key'              => 'frontend_section_settings',
                'route'            => 'dashboard.admin.frontend.sectionsettings',
                'label'            => 'Frontend Section Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 49,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.sectionsettings',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'frontend_menu' => [
                'parent_key'       => 'frontend',
                'key'              => 'frontend_menu',
                'route'            => 'dashboard.admin.frontend.menusettings',
                'label'            => 'Menu',
                'icon'             => null,
                'svg'              => null,
                'order'            => 50,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.menusettings',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'social_media_accounts' => [
                'parent_key'       => 'frontend',
                'key'              => 'social_media_accounts',
                'route'            => 'dashboard.admin.frontend.socialmedia',
                'label'            => 'Social Media Accounts',
                'icon'             => null,
                'svg'              => null,
                'order'            => 50,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.socialmedia',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'auth_settings' => [
                'parent_key'       => 'frontend',
                'key'              => 'auth_settings',
                'route'            => 'dashboard.admin.frontend.authsettings',
                'label'            => 'Auth Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 52,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.authsettings',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'f_a_q' => [
                'parent_key'       => 'frontend',
                'key'              => 'f_a_q',
                'route'            => 'dashboard.admin.frontend.faq.index',
                'label'            => 'F.A.Q',
                'icon'             => null,
                'svg'              => null,
                'order'            => 53,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.faq.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'tools_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'tools_section',
                'route'            => 'dashboard.admin.frontend.tools.index',
                'label'            => 'Tools Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 54,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.tools.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'features_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'features_section',
                'route'            => 'dashboard.admin.frontend.future.index',
                'label'            => 'Features Section',
                'icon'             => 'tabler-list-details',
                'svg'              => null,
                'order'            => 55,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.future.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'testimonials_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'testimonials_section',
                'route'            => 'dashboard.admin.testimonials.index',
                'label'            => 'Testimonials Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 56,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.testimonials.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'clients_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'clients_section',
                'route'            => 'dashboard.admin.clients.index',
                'label'            => 'Clients Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 57,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.clients.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'how_it_works_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'how_it_works_section',
                'route'            => 'dashboard.admin.howitWorks.index',
                'label'            => 'How it Works Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 58,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.howitWorks.*'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'who_can_use_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'who_can_use_section',
                'route'            => 'dashboard.admin.frontend.whois.index',
                'label'            => 'Who Can Use Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 59,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.whois.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'generators_list_section' => [
                'parent_key'       => 'frontend',
                'key'              => 'generators_list_section',
                'route'            => 'dashboard.admin.frontend.generatorlist.index',
                'label'            => 'Generators List Section',
                'icon'             => null,
                'svg'              => null,
                'order'            => 60,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.frontend.generatorlist.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'finance' => [
                'parent_key'       => null,
                'key'              => 'finance',
                'route'            => 'dashboard.admin.finance.plans.index',
                'label'            => 'Finance',
                'icon'             => 'tabler-wallet',
                'svg'              => null,
                'order'            => 61,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.*', 'dashboard.admin.bank.transactions.list',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'bank_transactions' => [
                'parent_key'       => 'finance',
                'key'              => 'bank_transactions',
                'route'            => 'dashboard.admin.bank.transactions.list',
                'label'            => 'Bank Transactions',
                'icon'             => null,
                'svg'              => null,
                'order'            => 62,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.bank.transactions.list',
                ],
                'show_condition' => $admin && bankActive(),
            ],
            'membership_plans' => [
                'parent_key'       => 'finance',
                'key'              => 'membership_plans',
                'route'            => 'dashboard.admin.finance.plans.index',
                'label'            => 'Membership Plans (old version)',
                'icon'             => null,
                'svg'              => null,
                'order'            => 63,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.plans.*',
                ],
                'show_condition' => false,
                'is_admin'       => true,
            ],
            'admin_finance_plan' => [
                'parent_key'       => 'finance',
                'key'              => 'admin_finance_plan',
                'route'            => 'dashboard.admin.finance.plan.index',
                'label'            => 'Pricing Plans',
                'icon'             => null,
                'svg'              => null,
                'order'            => 63,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.plan.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
                'badge'          => 'new',
            ],
            'payment_gateways' => [
                'parent_key'       => 'finance',
                'key'              => 'payment_gateways',
                'route'            => 'dashboard.admin.finance.paymentGateways.index',
                'label'            => 'Payment Gateways',
                'icon'             => null,
                'svg'              => null,
                'order'            => 64,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.paymentGateways.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'trial_features' => [
                'parent_key'       => 'finance',
                'key'              => 'trial_features',
                'route'            => 'dashboard.admin.finance.free.feature',
                'label'            => 'Trial Features',
                'icon'             => null,
                'svg'              => null,
                'order'            => 60,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.free.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'api_cost_management' => [
                'parent_key'       => 'finance',
                'key'              => 'api_cost_management',
                'route'            => 'dashboard.admin.finance.api-cost-management.index',
                'label'            => 'API Cost Management',
                'icon'             => null,
                'svg'              => null,
                'order'            => 60,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.api-cost-management.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'mobile_payment' => [
                'parent_key'       => 'finance',
                'key'              => 'mobile_payment',
                'route'            => 'dashboard.admin.finance.mobile.index',
                'label'            => 'Mobile Payment',
                'icon'             => null,
                'svg'              => null,
                'order'            => 60,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.finance.mobile.index',
                ],
                'show_condition' => Helper::setting('mobile_payment_active', null, $setting),
                'is_admin'       => $admin,
            ],
            'pages' => [
                'parent_key'       => null,
                'key'              => 'pages',
                'route'            => 'dashboard.page.list',
                'label'            => 'Pages',
                'icon'             => 'tabler-file-description',
                'svg'              => null,
                'order'            => 61,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.page.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'blog' => [
                'parent_key'       => null,
                'key'              => 'blog',
                'route'            => 'dashboard.blog.list',
                'label'            => 'Blog',
                'icon'             => 'tabler-pencil',
                'svg'              => null,
                'order'            => 62,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.blog.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'affiliates_admin' => [
                'parent_key'       => null,
                'key'              => 'affiliates_admin',
                'route'            => 'dashboard.admin.affiliates.index',
                'label'            => 'Affiliates',
                'icon'             => 'tabler-currency-dollar',
                'svg'              => null,
                'order'            => 63,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.affiliates.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'coupons_admin' => [
                'parent_key'       => null,
                'key'              => 'coupons_admin',
                'route'            => 'dashboard.admin.coupons.index',
                'label'            => 'Coupons',
                'icon'             => 'tabler-ticket',
                'svg'              => null,
                'order'            => 64,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.coupons.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'email_templates' => [
                'parent_key'       => null,
                'key'              => 'email_templates',
                'route'            => 'dashboard.email-templates.index',
                'label'            => 'Email Templates',
                'icon'             => 'tabler-mail',
                'svg'              => null,
                'order'            => 65,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.email-templates.*',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'introductions' => [
                'parent_key'       => null,
                'key'              => 'introductions',
                'route'            => 'dashboard.introductions.index',
                'label'            => 'User Onboarding',
                'icon'             => 'tabler-directions',
                'svg'              => null,
                'order'            => 65,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.introductions.*',
                ],
                'show_condition' => Route::has('dashboard.introductions.index'),
                'is_admin'       => true,
            ],
            'mailchimp_newsletter' => [
                'parent_key'       => null,
                'key'              => 'mailchimp_newsletter',
                'route'            => 'dashboard.admin.mailchimp-newsletter.index',
                'label'            => 'Mailchimp Newsletter',
                'icon'             => 'tabler-mailbox',
                'svg'              => null,
                'order'            => 65,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.mailchimp-newsletter.*',
                ],
                'show_condition' => Route::has('dashboard.admin.mailchimp-newsletter.index'),
                'is_admin'       => true,
            ],
            'hubspot' => [
                'parent_key'       => null,
                'key'              => 'hubspot',
                'route'            => 'dashboard.admin.hubspot.index',
                'label'            => 'Hubspot',
                'icon'             => 'tabler-affiliate',
                'svg'              => null,
                'order'            => 65,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.hubspot.*',
                ],
                'show_condition' => Route::has('dashboard.admin.hubspot.index'),
                'is_admin'       => true,
            ],
            'api_integration' => [
                'parent_key'       => null,
                'key'              => 'api_integration',
                'route'            => 'default',
                'label'            => 'API Integration',
                'icon'             => 'tabler-api',
                'svg'              => null,
                'order'            => 66,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.settings.openai', 'dashboard.admin.settings.gemini',
                    'dashboard.admin.settings.anthropic', 'dashboard.admin.settings.fal-ai',
                    'dashboard.admin.settings.stablediffusion', 'dashboard.admin.settings.unsplashapi',
                    'dashboard.admin.settings.pexelsapi', 'dashboard.admin.settings.pixabayapi',
                    'dashboard.admin.settings.serperapi',
                    'dashboard.admin.settings.tts', 'dashboard.admin.settings.synthesia',
                    'dashboard.admin.settings.aimlapi', 'dashboard.admin.settings.pebblely',
                    'dashboard.admin.settings.plagiarism', 'dashboard.admin.settings.clipdrop', 'dashboard.admin.settings.heygen',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'api_integration_openai' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_openai',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.openai',
                'label'            => 'OpenAI',
                'icon'             => null,
                'svg'              => null,
                'order'            => 67,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.openai'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_gemini' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_gemini',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.gemini',
                'label'            => 'Gemini',
                'icon'             => null,
                'svg'              => null,
                'order'            => 67,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.gemini.*'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_anthropic' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_anthropic',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.anthropic',
                'label'            => 'Anthropic',
                'icon'             => null,
                'svg'              => null,
                'order'            => 68,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.anthropic'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_fal_ai' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_fal_ai',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.fal-ai',
                'label'            => EngineEnum::FAL_AI->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 68,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.fal-ai'],
                'show_condition'   => Route::has('dashboard.admin.settings.fal-ai'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
                'badge'            => 'new',
            ],
            'api_integration_stablediffusion' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_stablediffusion',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.stablediffusion',
                'label'            => 'StableDiffusion',
                'icon'             => null,
                'svg'              => null,
                'order'            => 69,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.stablediffusion'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_unsplashapi' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_unsplashapi',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.unsplashapi',
                'label'            => EngineEnum::UNSPLASH->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 70,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.unsplashapi'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_pexelsapi' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_pexelsapi',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.pexelsapi',
                'label'            => EngineEnum::PEXELS->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 71,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.pexelsapi'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_pixabayapi' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_pixabayapi',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.pixabayapi',
                'label'            => EngineEnum::PIXABAY->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 72,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.pixabayapi'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_serperapi' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_serperapi',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.serperapi',
                'label'            => EngineEnum::SERPER->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 73,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.serperapi'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_tts' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_tts',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.tts',
                'label'            => 'TTS',
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.tts'],
                'show_condition'   => true,
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_synthesia' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_synthesia',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.synthesia',
                'label'            => EngineEnum::SYNTHESIA->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.synthesia'],
                'show_condition'   => Route::has('dashboard.user.ai-avatar.index'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_heygen' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_heygen',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.heygen',
                'label'            => EngineEnum::HEYGEN->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.heygen'],
                'show_condition'   => Route::has('dashboard.user.ai-avatar-pro.index'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'api_integration_pebblely' => [
                'parent_key'       => 'api_integration',
                'key'              => 'api_integration_pebblely',
                'route'            => Helper::appIsDemo() ? 'default' : 'dashboard.admin.settings.pebblely',
                'label'            => EngineEnum::PEBBLELY->label(),
                'icon'             => null,
                'svg'              => null,
                'order'            => 74,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.admin.settings.pebblely'],
                'show_condition'   => Route::has('dashboard.user.ai-product-shot.index'),
                'is_admin'         => true,
                'onclick'          => Helper::appIsDemo() ? 'return toastr.info(\'This feature is disabled in Demo version.\')' : '',
            ],
            'plagiarism_extension' => [
                'parent_key'       => 'api_integration',
                'key'              => 'plagiarism_extension',
                'route'            => 'dashboard.admin.settings.plagiarism',
                'label'            => 'Plagiarism API',
                'icon'             => null,
                'svg'              => null,
                'order'            => 77,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.plagiarism',
                ],
                'show_condition' => $admin && Route::has('dashboard.admin.settings.plagiarism'),
            ],
            'settings' => [
                'parent_key'       => null,
                'key'              => 'settings',
                'route'            => 'dashboard.admin.settings.general',
                'label'            => 'Settings',
                'icon'             => 'tabler-device-laptop',
                'svg'              => null,
                'order'            => 75,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => [
                    'dashboard.admin.settings.*', 'elseyyid.translations.home',
                    'elseyyid.translations.lang', 'dashboard.admin.config.*', 'dashboard.admin.ai-chat-model.index',
                ],
                'show_condition' => true,
                'is_admin'       => true,
            ],
            'config' => [
                'parent_key'       => 'settings',
                'key'              => 'config',
                'route'            => 'dashboard.admin.config.index',
                'label'            => 'General Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 0,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.config.index'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'thumbnail_system' => [
                'parent_key'       => 'settings',
                'key'              => 'thumbnail_system',
                'route'            => 'dashboard.admin.settings.thumbnail',
                'label'            => 'Thumbnail System',
                'icon'             => null,
                'svg'              => null,
                'order'            => 79,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.thumbnail'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'privacy' => [
                'parent_key'       => 'settings',
                'key'              => 'privacy',
                'route'            => 'dashboard.admin.settings.privacy',
                'label'            => 'Privacy Policy and Terms',
                'icon'             => null,
                'svg'              => null,
                'order'            => 82,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.settings.privacy'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'site_health' => [
                'parent_key'       => null,
                'key'              => 'site_health',
                'route'            => 'dashboard.admin.health.index',
                'label'            => 'Site Health',
                'icon'             => 'tabler-activity-heartbeat',
                'svg'              => null,
                'order'            => 85,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.health.index'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'license' => [
                'parent_key'       => null,
                'key'              => 'license',
                'route'            => 'dashboard.admin.license.index',
                'label'            => 'License',
                'icon'             => 'tabler-checklist',
                'svg'              => null,
                'order'            => 86,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.license.index'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'update' => [
                'parent_key'       => null,
                'class'            => 'nav-link--update',
                'key'              => 'update',
                'route'            => 'dashboard.admin.update.index',
                'label'            => 'Update',
                'icon'             => 'tabler-refresh',
                'svg'              => null,
                'order'            => 87,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.update.index'],
                'show_condition'   => true,
                'is_admin'         => true,
            ],
            'menu_setting' => [
                'parent_key'       => null,
                'class'            => '',
                'key'              => 'menu_setting',
                'route'            => 'dashboard.admin.menu.index',
                'label'            => 'Menu',
                'icon'             => 'tabler-menu',
                'svg'              => null,
                'order'            => 88,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => null,
                'active_condition' => ['dashboard.admin.menu.index'],
                'show_condition'   => Route::has('dashboard.admin.menu.index'),
                'is_admin'         => true,
            ],
        ];

        return $this->mergeExtensionMenu($menu);
    }

    public function mergeExtensionMenu($menu)
    {
        // automation extension
        $is_automation_active = false;

        if (class_exists('App\Extensions\AISocialMedia\System\Models\Automation')) {
            $is_automation_active = \App\Extensions\AISocialMedia\System\Models\Automation::getAutomationActive();
        }

        $menu = array_merge($menu, [
            'openai_generator_extension' => [
                'parent_key'       => null,
                'key'              => 'openai_generator_extension',
                'route'            => 'default',
                'label'            => 'Ai Template',
                'icon'             => 'tabler-list-details',
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.ai-template.openai-generator.*', 'dashboard.user.ai-template.openai-generator-filter.*',
                ],
                'show_condition' => Route::has('dashboard.user.ai-template.openai-generator.index') && Route::has('dashboard.user.ai-template.openai-generator-filter.index') && setting('ai_template_for_customer', '1') == '1',
            ],
            'custom_templates_extension' => [
                'parent_key'       => 'openai_generator_extension',
                'key'              => 'custom_templates_extension',
                'route'            => 'dashboard.user.ai-template.openai-generator.index',
                'label'            => 'Custom Templates',
                'icon'             => null,
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.ai-template.openai-generator.*',
                ],
                'show_condition' => Route::has('dashboard.user.ai-template.openai-generator.index') && Route::has('dashboard.user.ai-template.openai-generator-filter.index') && setting('ai_template_for_customer', '1') == '1',
            ],
            'ai_writer_categories_extension' => [
                'parent_key'       => 'openai_generator_extension',
                'key'              => 'ai_writer_categories_extension',
                'route'            => 'dashboard.user.ai-template.openai-generator-filter.index',
                'label'            => 'AI Writer Categories',
                'icon'             => null,
                'svg'              => null,
                'order'            => 5,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => ['dashboard.user.ai-template.openai-generator-filter.*'],
                'show_condition'   => Route::has('dashboard.user.ai-template.openai-generator.index') && Route::has('dashboard.user.ai-template.openai-generator-filter.index') && setting('ai_template_for_customer', '1') == '1',
            ],
            'ai_plagiarism_extension' => [
                'parent_key'       => null,
                'key'              => 'ai_plagiarism_extension',
                'route'            => 'dashboard.user.openai.plagiarism.index',
                'label'            => 'AI Plagiarism',
                'icon'             => 'tabler-progress-check',
                'svg'              => null,
                'order'            => 6,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.openai.plagiarism.index',
                ],
                'show_condition' => Route::has('dashboard.user.openai.plagiarism.index'),
            ],
            'ai_detector_extension' => [
                'parent_key'       => null,
                'key'              => 'ai_detector_extension',
                'route'            => 'dashboard.user.openai.detectaicontent.index',
                'label'            => 'AI Detector',
                'icon'             => 'tabler-text-scan-2',
                'svg'              => null,
                'order'            => 6,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.openai.detectaicontent.index',
                ],
                'show_condition' => Route::has('dashboard.user.openai.detectaicontent.index'),
            ],
            'ai_social_media_extension' => [
                'parent_key'       => null,
                'key'              => 'ai_social_media_extension',
                'route'            => 'dashboard.user.automation.index',
                'label'            => 'AI Social Media',
                'icon'             => 'tabler-share',
                'svg'              => null,
                'order'            => 6,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.automation.index',
                ],
                'show_condition' => Route::has('dashboard.user.automation.index') && $is_automation_active,
            ],
            'scheduled_posts_extension' => [
                'parent_key'       => null,
                'key'              => 'scheduled_posts_extension',
                'route'            => 'dashboard.user.automation.list',
                'label'            => 'Scheduled Posts',
                'icon'             => 'tabler-report',
                'svg'              => null,
                'order'            => 6,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.automation.list',
                ],
                'show_condition' => Route::has('dashboard.user.automation.list') && $is_automation_active,
            ],
            'ai_social_media_settings_extension' => [
                'parent_key'       => 'settings',
                'key'              => 'ai_social_media_settings_extension',
                'route'            => 'dashboard.admin.automation.settings',
                'label'            => 'AI Social Media Settings',
                'icon'             => null,
                'svg'              => null,
                'order'            => 77,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.automation.settings',
                ],
                'show_condition' => Route::has('dashboard.admin.automation.settings'),
            ],

            'chat_settings_extension' => [
                'parent_key'       => null,
                'key'              => 'chat_settings_extension',
                'route'            => 'default',
                'label'            => 'Chat Settings',
                'icon'             => 'tabler-message-circle',
                'svg'              => null,
                'order'            => 7,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.chat-setting.chat-category.*', 'dashboard.user.chat-setting.chatbot.*', 'dashboard.user.chat-setting.chat-template.*',
                ],
                'show_condition' => Route::has('dashboard.user.chat-setting.chat-category.index') && (setting('chat_setting_for_customer', 0) == 1),
            ],
            'chat_categories_extension' => [
                'parent_key'       => 'chat_settings_extension',
                'key'              => 'chat_categories_extension',
                'route'            => 'dashboard.user.chat-setting.chat-category.index',
                'label'            => 'Chat Categories',
                'icon'             => null,
                'svg'              => null,
                'order'            => 1,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.chat-setting.chat-category.*',
                ],
                'show_condition' => Route::has('dashboard.user.chat-setting.chat-category.index') && (setting('chat_setting_for_customer', 0) == 1),
            ],
            'chat_template_extension' => [
                'parent_key'       => 'chat_settings_extension',
                'key'              => 'chat_template_extension',
                'route'            => 'dashboard.user.chat-setting.chat-template.index',
                'label'            => 'Chat Templates',
                'icon'             => null,
                'svg'              => null,
                'order'            => 2,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.chat-setting.chat-template.*',
                ],
                'show_condition' => Route::has('dashboard.user.chat-setting.chat-template.index') && (setting('chat_setting_for_customer', '0') == 1),
            ],
            'chat_training_extension' => [
                'parent_key'       => 'chat_settings_extension',
                'key'              => 'chat_training_extension',
                'route'            => 'dashboard.user.chat-setting.chatbot.index',
                'label'            => 'Chatbot Training',
                'icon'             => null,
                'svg'              => null,
                'order'            => 3,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.chat-setting.chatbot.*',
                ],
                'show_condition' => Route::has('dashboard.user.chat-setting.chatbot.index') && (setting('chat_setting_for_customer', '0') == 1),
            ],
            'cloudflare_r2_extension' => [
                'parent_key'       => 'settings',
                'key'              => 'cloudflare_r2_extension',
                'route'            => 'dashboard.admin.settings.cloudflare-r2',
                'label'            => 'Cloudflare R2',
                'icon'             => null,
                'svg'              => null,
                'order'            => 8,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.cloudflare-r2',
                ],
                'show_condition' => Route::has('dashboard.admin.settings.cloudflare-r2'),
            ],
            'photo_studio_extension' => [
                'parent_key'       => null,
                'key'              => 'photo_studio_extension',
                'route'            => 'dashboard.user.photo-studio.index',
                'label'            => 'AI Photo Studio',
                'icon'             => 'tabler-device-laptop',
                'svg'              => null,
                'order'            => 8,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.photo-studio.index',
                ],
                'show_condition' => Route::has('dashboard.user.photo-studio.index') && setting('photo_studio', 1) == 1,
            ],
            'seo_tool_extension' => [
                'parent_key'       => null,
                'key'              => 'seo_tool_extension',
                'route'            => 'dashboard.user.seo.index',
                'label'            => 'SEO Tool',
                'icon'             => 'tabler-seo',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.seo.index',
                ],
                'show_condition' => Route::has('dashboard.user.seo.index') && setting('seo_ai_tool', 1) == 1,
            ],
            'ai_web_chat_extension' => [
                'parent_key'       => null,
                'key'              => 'ai_web_chat_extension',
                'route'            => 'dashboard.user.openai.webchat.workbook',
                'label'            => 'AI Web Chat',
                'icon'             => 'tabler-world-www',
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.user.openai.webchat.workbook',
                ],
                'show_condition' => Route::has('dashboard.user.openai.webchat.workbook'),
            ],
            'photo_studio_setting' => [
                'parent_key'       => 'api_integration',
                'key'              => 'photo_studio_setting',
                'route'            => 'dashboard.admin.settings.clipdrop',
                'label'            => 'Photo Studio',
                'icon'             => null,
                'svg'              => null,
                'order'            => 10,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.clipdrop',
                ],
                'show_condition' => Route::has('dashboard.user.photo-studio.index') && Route::has('dashboard.admin.settings.clipdrop'),
            ],
            'maintenance_setting' => [
                'parent_key'       => 'settings',
                'key'              => 'maintenance_setting',
                'route'            => 'dashboard.admin.settings.maintenance.index',
                'label'            => 'Maintenance',
                'icon'             => null,
                'svg'              => null,
                'order'            => 8,
                'is_active'        => true,
                'params'           => [],
                'type'             => 'item',
                'is_admin'         => true,
                'extension'        => true,
                'active_condition' => [
                    'dashboard.admin.settings.maintenance.index',
                ],
                'show_condition' => Route::has('dashboard.admin.settings.maintenance.index'),
            ],
        ]);

        return $menu;
    }

    public function extensionCheck(string $slug): bool
    {
        return Extension::query()->where('slug', $slug)->where('installed', true)->exists();
    }

    public function showTeamFunctionality(): bool
    {
        $checkPlan = Plan::query()->where('is_team_plan', 1)->first();

        return Helper::setting('team_functionality') && ! auth()?->user()?->getAttribute('team_id') && $checkPlan;
    }

    /**
     * For new plan ai tools
     */
    public static function planAiToolsMenu(): array
    {
        $keys = [
            'ai_writer', 'ai_chat_all', 'ai_image_generator', 'ai_video', 'seo_tool_extension', 'ai_voiceover',
            'ai_pdf', 'ai_vision', 'ai_speech_to_text', 'photo_studio_extension', 'ai_rewriter', 'ai_editor',
            'ai_code_generator', 'ai_youtube', 'ai_chat_image', 'ai_rss', 'ai_voiceover_clone', 'ai_web_chat_extension',
            'ai_social_media_extension', 'ai_detector_extension', 'ai_plagiarism_extension', 'ai_article_wizard', 'ai_voice_isolator', 'ext_chat_bot',
        ];

        $data = (new self)->generate();

        foreach ($data as $datum) {
            $children = data_get($datum, 'children');

            if (is_array($children)) {

                $data = array_merge($data, $children);
            }
        }

        return collect($data)
            ->filter(function ($item) use ($keys) {
                if (! isset($item['show_condition'])) {
                    $item['show_condition'] = false;
                }

                return in_array($item['key'], $keys) && $item['show_condition'];
            })
            ->values()
            ->toArray();
    }

    public static function planFeatureMenu(): array
    {
        $data = (new self)->generate();

        foreach ($data as $datum) {
            $children = data_get($datum, 'children');

            if (is_array($children)) {

                $data = array_merge($data, $children);
            }
        }

        return collect($data)

            ->filter(function ($item) {
                if (! isset($item['show_condition'])) {
                    $item['show_condition'] = false;
                }

                return in_array($item['key'], [
                    'api_keys',
                    'brand_voice',
                    'support',
                    'integration',
                    'custom_templates_extension',
                    'chat_training_extension',
                ]) && $item['show_condition'];
            })
            ->values()
            ->map(function ($item) {
                if ($item['key'] === 'api_keys') {
                    $item['label'] = 'Personal API Key';
                }
                if ($item['key'] === 'integration') {
                    $item['label'] = 'WordPress';
                }

                return $item;
            })
            ->toArray();
    }
}
